% VignettingCorrection does vignetting correction, white balancing and
% debayering for a set of plenoptic images
%
%   This function uses a recorded white image of the plenoptic camera to
%   do vignetting correction and white balancing for a set of images of the
%   plenoptic camera. Afterwards debayering is performed and image are
%   converted to grayscale.
%
%   The processed images are stored at '$(pathRawImages)_processed'.
%   Furthermore, if a set of white images is given instead of a single
%   image, the function calculates a avarage white image and stores it as
%   'AverageWhiteImage.png'. This white image is then used for correction.
%
%   The function VignettingCorrection can be called as follows:
%   VignettingCorrection(pathRawImages, whiteImage, outputFormat)
%
%   Inputs:
%       pathRawImages = path to folder containing images
%       whiteImage = path to white image or path to folder containing set
%                    of white images
%       outputFormat = output format ('.png' or '.jpg')
%
%   Outputs:
%       none
%
%   Author:         Niclas Zeller
%   Date:           2018-06-23

function VignettingCorrection(pathRawImages, whiteImage, outputFormat)

%% calculate avarage white image
if(isdir(whiteImage))
    % if white image is a directory calculate average white image
    if(whiteImage(end)=='/' || whiteImage(end)=='\')
        whiteImage = whiteImage(1:end-1);
    end
    whiteImageList = dir([whiteImage '/*.png']);
    
    if(isempty(whiteImageList))
        error('did not find any png files in %s.',whiteImage)
    end
    
    avWhiteImage = im2double(imread([whiteImage '/' whiteImageList(1).name]));
    [~,~,ch] = size(avWhiteImage);
    if(ch~=1)
        error('image read is not a monochromatic image.')
    end
    for idx=2:length(whiteImageList)
        avWhiteImage = avWhiteImage + im2double(imread([whiteImage '/' whiteImageList(idx).name]));
    end
    avWhiteImage = avWhiteImage/max(max(avWhiteImage));
    imwrite(uint16(avWhiteImage*(2^16-1)),[whiteImage '/../AverageWhiteImage.png']);
    whiteImage = [whiteImage '/../AverageWhiteImage.png'];  
end

%% process raw image sequence
if(pathRawImages(end)=='/' || pathRawImages(end)=='\')
    pathRawImages = pathRawImages(1:end-1);
end

if(~isdir(pathRawImages))
    error('raw image path (%s) does not exist',pathRawImages)
end

% get raw image list
rawImageList = dir([pathRawImages '/*.png']);
if(isempty(rawImageList))
    error('did not find any png files in %s.',pathRawImages)
end

% load white image
[~,~,ext] = fileparts(whiteImage);
if(~strcmp(ext,'.png'))
    error('given white image has the wrong file format. (must be .png)')
elseif(~exist(whiteImage,'file'))
    error('given white image (%s) does not exist.',whiteImage)
end

avWhiteImage = im2double(imread(whiteImage));
[~,~,ch] = size(avWhiteImage);
if(ch~=1)
    error('white image read is not a monochromatic image.')
end

% process images
if(~strcmp(outputFormat,'.jpg') && ~strcmp(outputFormat,'.png'))
    error('invalid output format give');
end

time_stamp_file = dir([pathRawImages '/*.txt']);
if(length(time_stamp_file)~=1)
    warning('no time stamp file found')
    numFiles = -1;
else
    time_stamp_file = time_stamp_file.name;
    file = fopen([pathRawImages '/' time_stamp_file]);
    data = textscan(file,'%s %f %f');
    fileName = data{1};
    timeStamp = data{2};
    exposure = data{3};
    numFiles = length(timeStamp);
    fclose(file);
    clear data;

    if(length(rawImageList) ~= numFiles)
        error('number of image files does not match time stamp list')
    end
end

pathRawImagesOut = [pathRawImages '_processed'];
mkdir(pathRawImagesOut);

if(numFiles>0)
    fileOut = fopen([pathRawImagesOut '/' time_stamp_file],'w+');
end

for idx = 1 : length(rawImageList)
    fprintf('processing %s\n',rawImageList(idx).name);
    % vignietting correction
    rawImage = double(imread([pathRawImages '/' rawImageList(idx).name]));
    correctedRawImage = uint8(rawImage./avWhiteImage);
    
    % debayering
    processedImage = rgb2gray(demosaic(correctedRawImage,'gbrg'));
    [~,name,~] = fileparts(rawImageList(idx).name);
    imwrite(processedImage,[pathRawImagesOut '/' name outputFormat]);

    if(numFiles>0)
        fprintf(fileOut,'%s %d %d\n',[name outputFormat], timeStamp(idx), exposure(idx));
    end
end

if(numFiles>0)
    fclose(fileOut);
end

end
