% RectStereoImages does rectification and vignetting correction for a set
% of stereo image pairs.
%
%   This function uses a given stereo camera model to calculate rectified
%   stereo image pairs. Futhermore recorded white images are used to
%   perform vignetting correction.
%
%   If sets of white images is given instead of single images, the function
%   calculates a avarage white images and stores them as 
%   'AverageWhiteImage_cam0.png' and 'AverageWhiteImage_cam1.png'. Theres
%   white images are then used for correction.
%
%   The function stores a file 'xxxx_rect.txt' which contains the rectified
%   stereo camera parameters. Furthermore a file called 'camera.txt' is
%   created which contains the camera paramters of one rectified camera
%   (either left or right camera) in a format compatible to the open-source
%   implementations of LSD-SLAM and DSO. A file 'assoc_stereo.txt' is
%   created containing the image times stamps compatible to open-source
%   implementation of ORB-SLAM2.
%
%   The rectified images are stored at '$(path_cam0)_rect' and
%   '$(path_cam1)_rect' respectively
%
%   The function RectStereoImages can be called as follows:
%   RectStereoImages(cameraModel, outputFormat, path_cam0, path_cam1, ...
%   whiteImage_cam0, whiteImage_cam1)
%
%   Inputs:
%       cameraModel = path to the stereo camera model file
%       outputFormat = output format ('.png' or '.jpg')
%       path_cam0 = path to folder containing left camera images
%       path_cam1 = path to folder containing right camera images
%       whiteImage_cam0 = path to white image of left camera or path to
%                         folder containing set of white images
%       whiteImage_cam1 = path to white image of right camera or path to
%                         folder containing set of white images
%
%   Outputs:
%       none
%
%   Author:         Niclas Zeller
%   Date:           2018-06-23

function RectStereoImages(cameraModel, outputFormat, path_cam0, path_cam1, whiteImage_cam0, whiteImage_cam1)

%% check if camera model does exist
[filePathCamModel,fileNameCamModel,extCamModel] = fileparts(cameraModel);
if(~strcmp(extCamModel,'.xml'))
    error('The given camera model file has the wrong extension.')
end
if(~exist(cameraModel,'file'))
    error('The given camera model file does not exist.')
end

%% check if given directories exist
if(~exist(path_cam0,'dir'))
    error('The first given directory does not exist.')
end

if(path_cam0(end) == '/' || path_cam0(end)=='\')
    path_cam0 = path_cam0(1:end-1);
end
path_cam0_dst = [path_cam0 '_rect'];

if(~exist(path_cam1,'dir'))
    error('The second given directory does not exist.')
end
if(path_cam1(end) == '/' || path_cam1(end)=='\')
    path_cam1 = path_cam1(1:end-1);
end
path_cam1_dst = [path_cam1 '_rect'];

%% check if output format is valid
if(~strcmp(outputFormat,'.jpg') && ~strcmp(outputFormat,'.png'))
    error('invalid output format give');
end

%% read in camera model
stereoCameraStruct = Xml2Struct(cameraModel);
stereoCameraStruct = stereoCameraStruct.Root;
if(~strcmp(stereoCameraStruct.CalibrationModel,'StereoCamera'))
    error('camera model %s has the wrong format.',cameraModel)
end

% create cameraParameters object for camera 0
intinsicmatrix = eye(3);
intinsicmatrix(1,1) = stereoCameraStruct.Camera_0.fx;
intinsicmatrix(2,2) = stereoCameraStruct.Camera_0.fy;
intinsicmatrix(3,1) = stereoCameraStruct.Camera_0.cx+1;
intinsicmatrix(3,2) = stereoCameraStruct.Camera_0.cy+1;

radialDist = [stereoCameraStruct.Camera_0.RadialDistortion.A0, ...
    stereoCameraStruct.Camera_0.RadialDistortion.A1, ...
    stereoCameraStruct.Camera_0.RadialDistortion.A2];

if(isfield(stereoCameraStruct.Camera_0,'TangentialDistortion'))
    tangentDist = [stereoCameraStruct.Camera_0.TangentialDistortion.B0, ...
        stereoCameraStruct.Camera_0.TangentialDistortion.B1];
else
    tangentDist = [0, 0];
end

cam0 = cameraParameters('IntrinsicMatrix', intinsicmatrix, ...
         'RadialDistortion', radialDist, 'TangentialDistortion', tangentDist);
     
% create cameraParameters object for camera 1
intinsicmatrix = eye(3);
intinsicmatrix(1,1) = stereoCameraStruct.Camera_1.fx;
intinsicmatrix(2,2) = stereoCameraStruct.Camera_1.fy;
intinsicmatrix(3,1) = stereoCameraStruct.Camera_1.cx+1;
intinsicmatrix(3,2) = stereoCameraStruct.Camera_1.cy+1;

radialDist = [stereoCameraStruct.Camera_1.RadialDistortion.A0, ...
    stereoCameraStruct.Camera_1.RadialDistortion.A1, ...
    stereoCameraStruct.Camera_1.RadialDistortion.A2];

if(isfield(stereoCameraStruct.Camera_1,'TangentialDistortion'))
    tangentDist = [stereoCameraStruct.Camera_1.TangentialDistortion.B0, ...
        stereoCameraStruct.Camera_1.TangentialDistortion.B1];
else
    tangentDist = [0, 0];
end

cam1 = cameraParameters('IntrinsicMatrix', intinsicmatrix, ...
         'RadialDistortion', radialDist, 'TangentialDistortion', tangentDist);

% camera orientation
T = reshape(stereoCameraStruct.CameraOrientation.Coeff,[4,4]);
stereoParam = stereoParameters(cam0, cam1, T(1:3,1:3), T(4,1:3));

%% check if white image is given
if(isempty(whiteImage_cam0) || isempty(whiteImage_cam1))
    disp('No white images given.\n');
    correctVignietting = false;
else
    whiteImage_cam0 = VignettingCorrection(whiteImage_cam0,0);
    whiteImage_cam1 = VignettingCorrection(whiteImage_cam1,1);
    correctVignietting = true;
end

mkdir(path_cam0_dst)
mkdir(path_cam1_dst)

%% read file lists
fileListCam0 = dir([path_cam0 '/*.png']);
fileListCam0 = {fileListCam0.name}';

fileListCam1 = dir([path_cam1 '/*.png']);
fileListCam1 = {fileListCam1.name}';

% check if length of file lists match
if(length(fileListCam0) ~= length(fileListCam0))
    error('differemt mumber of image files')
end

%% create association file for orb slam
% read time stamp
f_timeStamps = fopen([path_cam0 '/timeStamps.txt'],'r');
if(f_timeStamps<0)
    warning('could not open time stamp file -> creating pseudo time stamps')
    timeStamps = 0:1/30:1/30*(length(fileListCam0)-1);
    numFiles = length(fileListCam0);
else
    data = textscan(f_timeStamps,'%s %f %f');
    timeStamps = data{2};
    numFiles = length(timeStamps);
    fclose(f_timeStamps);
    clear data;
end

if(length(fileListCam0) ~= numFiles)
    error('number of time stamps does not match with number of files')
end
dst_folder_cam0 = cell2mat(getfield( fliplr(regexp(path_cam0_dst,filesep,'split')), {1}));
dst_folder_cam1 = cell2mat(getfield( fliplr(regexp(path_cam1_dst,filesep,'split')), {1}));

f_timeStamps = fopen([path_cam0 '/../assoc_stereo.txt'],'w+');

%% process images
for idx=1:length(fileListCam0)
   
   fprintf('rectifying image %d of %d\n',idx,length(fileListCam0))
   
   file_l = [path_cam0 '/' fileListCam0{idx}];
   file_r = [path_cam1 '/' fileListCam1{idx}];
   
   [~,fileNameCam0,~] = fileparts(fileListCam0{idx});
   [~,fileNameCam1,~] = fileparts(fileListCam1{idx});
   
   file_l_out = [path_cam0_dst '/' fileNameCam0 outputFormat];
   file_r_out = [path_cam1_dst '/' fileNameCam1 outputFormat];
     
   iml = im2double(imread(file_l));
   imr = im2double(imread(file_r));
   
   if(idx==1)
       [height,width,~] = size(iml);
       imSize = [height,width];
       [~, ~, Q, ~, ~] = computeRectificationParameters(stereoParam, imSize, 'full');
       f_new = Q(4,3);
       cx_new = -Q(4,1);
       cy_new = -Q(4,2);
       
       cx_av = 0.5*(stereoParam.CameraParameters1.PrincipalPoint(1)+stereoParam.CameraParameters1.PrincipalPoint(1));
       cy_av = 0.5*(stereoParam.CameraParameters1.PrincipalPoint(2)+stereoParam.CameraParameters1.PrincipalPoint(2));
       
       crop_x = round(cx_new-cx_av);
       cx_new = cx_new-crop_x;
       crop_y = round(cy_new-cy_av);
       cy_new = cy_new-crop_y;
       
       fCamModel = fopen([filePathCamModel '/' fileNameCamModel '_rect.txt'],'w+');
       fprintf(fCamModel,'%1.10f %1.10f %1.10f %1.10f %1.10f\n',f_new, f_new, cx_new-1, cy_new-1, norm(T(4,1:3)));
       fclose(fCamModel);
       
       fCamModel = fopen([filePathCamModel '/' 'camera.txt'],'w+');
       fprintf(fCamModel,'%1.10f %1.10f %1.10f %1.10f %d\n',f_new/width, f_new/height, (cx_new-0.5)/width, (cy_new-0.5)/height, 0);
       fprintf(fCamModel,'%d %d\n',width, height);
       fprintf(fCamModel,'%1.10f %1.10f %1.10f %1.10f %d\n',f_new/width, f_new/height, (cx_new-0.5)/width, (cy_new-0.5)/height, 0);
       fprintf(fCamModel,'%d %d\n',960, 720);
       fclose(fCamModel);
   end
   
   if(correctVignietting)
       iml = iml./whiteImage_cam0;
       imr = imr./whiteImage_cam1;
   end
   
   [iml_out,imr_out] = rectifyStereoImages(iml,imr,stereoParam,'OutputView', 'full');
   iml_out = iml_out(crop_y+1:crop_y+height,crop_x+1:crop_x+width,:);
   imr_out = imr_out(crop_y+1:crop_y+height,crop_x+1:crop_x+width,:);
   
   fprintf(f_timeStamps,'%d %s %d %s\n',timeStamps(idx), [dst_folder_cam0 '/' fileNameCam0 outputFormat], timeStamps(idx), [dst_folder_cam1 '/' fileNameCam1 outputFormat]);
   
   imwrite(iml_out, file_l_out)
   imwrite(imr_out, file_r_out)
end

fclose(f_timeStamps);

end


function [whiteImage] = VignettingCorrection(whiteImagePath, camNum)

%% calculate avarage white image
if(isdir(whiteImagePath))
    % if white image is a directory calculate average white image
    if(whiteImagePath(end)=='/' || whiteImagePath(end)=='\')
        whiteImagePath = whiteImagePath(1:end-1);
    end
    whiteImageList = dir([whiteImagePath '/*.png']);
    
    if(isempty(whiteImageList))
        error('did not find any png files in %s.',whiteImagePath)
    end
    
    avWhiteImage = im2double(imread([whiteImagePath '/' whiteImageList(1).name]));
    [~,~,ch] = size(avWhiteImage);
    if(ch~=1)
        error('image read is not a monochromatic image.')
    end
    for idx=2:length(whiteImageList)
        avWhiteImage = avWhiteImage + im2double(imread([whiteImagePath '/' whiteImageList(idx).name]));
    end
    avWhiteImage = avWhiteImage/max(max(avWhiteImage));
    
    avWhiteImage = imgaussfilt(avWhiteImage,1);
    imwrite(uint16(avWhiteImage*(2^16-1)),[whiteImagePath '/../' sprintf('AverageWhiteImage_cam%d.png', camNum)]);
    whiteImagePath = [whiteImagePath '/../' sprintf('AverageWhiteImage_cam%d.png', camNum)];  
end

whiteImage = double(imread(whiteImagePath))/(2^16-1);

end
