% Xml2Struct reads out xml-files and stores the content in a matlab
% structure.
%
%   This function reads a xml-file with xmlread and coverts the complete
%   content of the xml-file into a matlab structure. I differentiates
%   between string and numbers.
%
%   The function Xml2Struct can be called as follows:
%   xmlStruct = Xml2Struct( filePath )
%
%   Inputs:
%       filePath = a string which contains the path to the xml-file
%
%   Outputs:
%       xmlStruct = structure which contains the data read from the
%                   xml-file
%
%   Author:         Niclas Zeller
%   Date:           2018-06-23

function [ xmlStruct ] = Xml2Struct( filePath )

xmlFile = xmlread(filePath);

[ ~, xmlStruct ] = readOutNode( xmlFile );

end

% readOutNode reads out the data from a node in the xml-file.
%
%   The functions call itself recursively and creates a tree containing the
%   data of the xml-file.
%
%   Inputs:
%       node = Node which is processed
%
%   Outputs:
%       nodeName = String containing the name of the node
%       nodeContent = structure or variable containing the content of the
%                     node
function [ nodeName, nodeContent ] = readOutNode( node )

digits = '0123456789.,-e';

nodeName = char(node.getNodeName);
nodeContent = char(node.getTextContent());
digitCnt=sum(ismember(nodeContent,digits));
if(digitCnt == length(nodeContent))
    nodeContent = str2double(nodeContent);
end

if(~strcmp(nodeName,'#text'))
    nextChild = node.getFirstChild();
    while(~isempty(nextChild))
        [subNodeName, subNodeContent] = readOutNode( nextChild );
        if(~isempty(subNodeContent))
            if(exist('contentBranch'))
                if(isfield(contentBranch,subNodeName))
                    contentBranch.(subNodeName)(end+1,1) = subNodeContent;
                else
                    contentBranch.(subNodeName) = subNodeContent;
                end
            else
                contentBranch.(subNodeName) = subNodeContent;
            end
        end
        nextChild = nextChild.getNextSibling();
    end
    if(exist('contentBranch'))
        nodeContent = contentBranch;
    end
    
    if(node.hasAttributes)
        theAttributes = node.getAttributes;
        numAttributes = theAttributes.getLength;
        for count = 1:numAttributes
          attrib = theAttributes.item(count-1);
          attributes(count).Name = char(attrib.getName);
          attributes(count).Value = char(attrib.getValue);
          
          switch(attributes(count).Name)
              case 'id'
                  nodeName = ...
                      [nodeName '_' num2str(attributes(count).Value,0)];
              case 'type'
                  nodeName = ...
                      [nodeName '_' char(attributes(count).Value)];
          end
        end
    end    
else
    %return an empty node
    nodeName = '';
    nodeContent = [];
end
 
end

